// ============================================================
// Background Service Worker
// Bridges Chrome extension <-> Native Messaging Host
// ============================================================

const NATIVE_HOST_NAME = "com.vyrox.posprinterconnector";

chrome.runtime.onInstalled.addListener((details) => {
  if (details.reason === "install") {
    chrome.tabs.create({
      url: chrome.runtime.getURL("welcome.html")
    });
  }
});

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.action === "print") {
    sendToNativeHost(message.data)
      .then((response) => {
        sendResponse({ success: true, response: response });
      })
      .catch((error) => {
        sendResponse({ success: false, error: error.message || String(error) });
      });
    return true;
  }

  if (message.action === "testConnection") {
    sendToNativeHost({ command: "ping" })
      .then((response) => {
        sendResponse({ success: true, response: response });
      })
      .catch((error) => {
        sendResponse({ success: false, error: error.message || String(error) });
      });
    return true;
  }

  if (message.action === "getStatus") {
    sendToNativeHost({ command: "status" })
      .then((response) => {
        sendResponse({ success: true, response: response });
      })
      .catch((error) => {
        sendResponse({ success: false, error: error.message || String(error) });
      });
    return true;
  }
});

let _nativeQueue = Promise.resolve();

function sendToNativeHost(data) {
  return new Promise((resolve, reject) => {
    _nativeQueue = _nativeQueue
      .catch(() => {})
      .then(() => {
        return _doSendToNativeHost(data).then(resolve, reject);
      });
  });
}

function _doSendToNativeHost(data) {
  return new Promise((resolve, reject) => {
    let attempts = 0;
    const maxAttempts = 6;

    function tryConnect() {
      attempts++;
      try {
        let port = chrome.runtime.connectNative(NATIVE_HOST_NAME);
        let responded = false;
        let settled = false;

        port.onMessage.addListener((response) => {
          if (settled) return;
          responded = true;
          settled = true;
          try { port.disconnect(); } catch (e) {}
          resolve(response);
        });

        port.onDisconnect.addListener(() => {
          if (settled) return;
          if (!responded) {
            let error = chrome.runtime.lastError;
            let errMsg = error ? error.message : "Native host disconnected without response";

            if (errMsg.indexOf("forbidden") !== -1 && attempts < maxAttempts) {
              let delay = Math.min(300 + (attempts * 300), 2000);
              setTimeout(tryConnect, delay);
              return;
            }

            settled = true;
            reject(new Error(errMsg));
          }
        });

        port.postMessage(data);

        setTimeout(() => {
          if (!settled) {
            settled = true;
            try { port.disconnect(); } catch (e) {}
            reject(new Error("Native host timeout (10s)"));
          }
        }, 10000);
      } catch (err) {
        if (String(err).indexOf("forbidden") !== -1 && attempts < maxAttempts) {
          let delay = Math.min(300 + (attempts * 300), 2000);
          setTimeout(tryConnect, delay);
          return;
        }
        reject(err);
      }
    }

    tryConnect();
  });
}