// ============================================================
// Content Script - VYROX POS Printer Connector
// Detects print content on pages and injects print button
// CSS is self-contained (injected into page via <style>)
// ============================================================

(function () {
  "use strict";

  // ---- Inject Styles ----
  const STYLES = `
    #vyrox-pos-printer-connector-fab {
      position: fixed;
      bottom: 24px;
      right: 24px;
      z-index: 999999;
      display: flex;
      flex-direction: column;
      align-items: center;
      gap: 6px;
      font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
    }

    .rp-fab-btn {
      width: 52px;
      height: 52px;
      border-radius: 14px;
      background: #1a1a2e;
      color: #ffffff;
      display: flex;
      align-items: center;
      justify-content: center;
      cursor: pointer;
      box-shadow: 0 4px 16px rgba(0,0,0,0.24), 0 1px 4px rgba(0,0,0,0.12);
      transition: transform 200ms ease, box-shadow 200ms ease, background 200ms ease;
      user-select: none;
    }

    .rp-fab-btn:hover {
      transform: translateY(-2px);
      box-shadow: 0 6px 24px rgba(0,0,0,0.32), 0 2px 6px rgba(0,0,0,0.16);
      background: #16213e;
    }

    .rp-fab-btn:active {
      transform: translateY(0);
      box-shadow: 0 2px 8px rgba(0,0,0,0.2);
    }

    .rp-fab-btn svg {
      width: 24px;
      height: 24px;
    }

    .rp-fab-status {
      font-size: 11px;
      font-weight: 600;
      padding: 2px 8px;
      border-radius: 6px;
      background: rgba(26,26,46,0.85);
      color: #a0a0b8;
      white-space: nowrap;
      transition: all 200ms ease;
    }

    .rp-status-ready   { color: #a0a0b8; }
    .rp-status-sending { color: #f0c040; background: rgba(240,192,64,0.15); }
    .rp-status-success { color: #40c080; background: rgba(64,192,128,0.15); }
    .rp-status-error   { color: #f06060; background: rgba(240,96,96,0.15); }

    .rp-toast {
      position: fixed;
      top: 20px;
      right: 20px;
      z-index: 9999999;
      display: flex;
      align-items: flex-start;
      gap: 12px;
      padding: 14px 16px;
      border-radius: 10px;
      background: #1a1a2e;
      color: #e0e0e8;
      box-shadow: 0 8px 32px rgba(0,0,0,0.28);
      font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
      font-size: 13px;
      max-width: 360px;
      animation: rpToastIn 300ms ease forwards;
    }

    .rp-toast-success { border-left: 3px solid #40c080; }
    .rp-toast-error   { border-left: 3px solid #f06060; }

    .rp-toast-icon { flex-shrink: 0; margin-top: 1px; }
    .rp-toast-success .rp-toast-icon { color: #40c080; }
    .rp-toast-error .rp-toast-icon   { color: #f06060; }

    .rp-toast-body  { flex: 1; }
    .rp-toast-title { font-weight: 700; font-size: 13px; margin-bottom: 3px; }
    .rp-toast-message { font-size: 12px; color: #a0a0b8; line-height: 1.4; }

    .rp-toast-close {
      flex-shrink: 0;
      background: none;
      border: none;
      color: #606078;
      cursor: pointer;
      padding: 2px;
      border-radius: 4px;
      transition: color 150ms ease, background 150ms ease;
      display: flex;
      align-items: center;
      justify-content: center;
    }

    .rp-toast-close:hover {
      color: #e0e0e8;
      background: rgba(255,255,255,0.08);
    }

    .rp-toast-fadeout { animation: rpToastOut 300ms ease forwards; }

    @keyframes rpToastIn {
      from { opacity: 0; transform: translateX(40px); }
      to   { opacity: 1; transform: translateX(0); }
    }

    @keyframes rpToastOut {
      from { opacity: 1; transform: translateX(0); }
      to   { opacity: 0; transform: translateX(40px); }
    }

    @media (prefers-reduced-motion: reduce) {
      .rp-fab-btn,
      .rp-fab-status,
      .rp-toast,
      .rp-toast-close,
      .rp-toast-fadeout {
        animation: none !important;
        transition: none !important;
      }
    }
  `;

  function injectStyles() {
    var style = document.createElement("style");
    style.textContent = STYLES;
    document.head.appendChild(style);
  }

  // ---- Configuration ----
  var VYROXPOSPRINTERCONNECTOR_SELECTORS = [
    "#vyrox_pos_printer_connector",
    "#vyrox-pos-printer-connector",
    "#vyrox_pos_printer_connector-content",
    "#print-area",
    ".vyrox_pos_printer_connector",
    ".vyrox-pos-printer-connector",
    ".vyrox_pos_printer_connector-content",
    ".print-area",
    '[data-vyrox-pos-printer-connector="true"]',
    "#vyrox_printer_connector",
    "#vyrox-printer-connector",
    "#vyrox_printer_connector-content",
    ".vyrox_printer_connector",
    ".vyrox-printer-connector",
    ".vyrox_printer_connector-content",
    '[data-vyrox-printer-connector="true"]',
  ];

  var EXISTING_PRINT_BTN_SELECTORS = [
    "#btn-print",
    "#print-btn",
    ".btn-print",
    ".print-btn",
    '[data-action="print"]',
    'button[onclick*="print"]',
  ];

  var floatingBtn = null;
  var statusIndicator = null;

  // ---- Find print content element on page ----
  function findPrintContentElement() {
    for (var i = 0; i < VYROXPOSPRINTERCONNECTOR_SELECTORS.length; i++) {
      var el = document.querySelector(VYROXPOSPRINTERCONNECTOR_SELECTORS[i]);
      if (el) return el;
    }
    return null;
  }

  // ---- Extract print content as formatted text ----
  function extractPrintContentText(element) {
    if (!element) return null;

    var rawAttr = element.getAttribute("data-vyrox-pos-printer-connector-raw");
    if (!rawAttr) rawAttr = element.getAttribute("data-vyrox-printer-connector-raw");
    if (rawAttr) return rawAttr;

    var rawEl =
      element.querySelector(".vyrox-pos-printer-connector-raw") ||
      element.querySelector(".vyrox-printer-connector-raw") ||
      element.querySelector("[data-raw-print-content]") ||
      element.querySelector("pre.print-content-text");
    if (rawEl) return rawEl.textContent || rawEl.value;

    return htmlToPrintContentText(element);
  }

  // ---- Convert HTML print content to formatted plain text ----
  function htmlToPrintContentText(element) {
    var lines = [];

    if (element.children.length === 0) {
      return element.textContent.trim();
    }

    function processNode(node, context) {
      if (node.nodeType === Node.TEXT_NODE) {
        var text = node.textContent.trim();
        if (text) {
          context.texts.push(text);
        }
        return;
      }

      if (node.nodeType !== Node.ELEMENT_NODE) return;

      var tag = node.tagName.toLowerCase();
      var cls = node.className ? node.className.toLowerCase() : "";
      var style = window.getComputedStyle(node);
      var textAlign = style.textAlign;
      var fontWeight = parseInt(style.fontWeight) || 0;
      var fontSize = parseFloat(style.fontSize) || 12;
      var prefix = "";

      if (
        textAlign === "center" ||
        cls.includes("center") ||
        cls.includes("text-center")
      ) {
        prefix += "##CENTER##";
      }

      if (
        tag === "strong" ||
        tag === "b" ||
        fontWeight >= 700 ||
        cls.includes("bold") ||
        cls.includes("font-bold") ||
        cls.includes("fw-bold")
      ) {
        prefix += "##BOLD##";
      }

      if (
        fontSize >= 20 ||
        tag === "h1" ||
        tag === "h2" ||
        cls.includes("big") ||
        cls.includes("large") ||
        cls.includes("title") ||
        cls.includes("total") ||
        cls.includes("text-xl") ||
        cls.includes("text-2xl")
      ) {
        prefix += "##BIG##";
      }

      if (tag === "hr") {
        lines.push("--------------------------------");
        return;
      }

      if (
        cls.includes("separator") ||
        cls.includes("divider") ||
        cls.includes("line")
      ) {
        var sepText = node.textContent.trim();
        if (!sepText || /^[-=_.]+$/.test(sepText)) {
          lines.push("--------------------------------");
          return;
        }
      }

      if (cls.includes("cut") || node.getAttribute("data-cut") === "true") {
        lines.push("##CUT##");
        return;
      }

      if (cls.includes("feed") || node.getAttribute("data-feed")) {
        var feedCount = node.getAttribute("data-feed") || "3";
        lines.push("##FEED:" + feedCount + "##");
        return;
      }

if (tag === "tr") {
        var cells = node.querySelectorAll("td, th");
        if (cells.length > 0) {
          var cellTexts = Array.from(cells).map(function (c) {
            return c.textContent.trim();
          });

          var lineWidth = 48;
          var printAreaEl = element.closest('[data-print-width]');
          if (!printAreaEl) printAreaEl = element;
          if (printAreaEl && printAreaEl.getAttribute('data-print-width')) {
            var pw = parseInt(printAreaEl.getAttribute('data-print-width'));
            if (pw > 0) lineWidth = pw;
          }

          if (cells.length === 2) {
            var left = cellTexts[0];
            var right = cellTexts[1];
            var maxLeft = lineWidth - right.length - 2;
            if (left.length > maxLeft && maxLeft > 3) {
              left = left.substring(0, maxLeft - 1) + ".";
            }
            var pad = Math.max(1, lineWidth - left.length - right.length);
            lines.push(prefix + left + " ".repeat(pad) + right);
          } else if (cells.length === 3) {
            var qty = cellTexts[0].padEnd(4);
            var item = cellTexts[1];
            var price = cellTexts[2];
            var maxItem = lineWidth - qty.length - price.length - 2;
            if (item.length > maxItem && maxItem > 3) {
              item = item.substring(0, maxItem - 1) + ".";
            }
            var pad3 = Math.max(1, lineWidth - qty.length - item.length - price.length);
            lines.push(prefix + qty + item + " ".repeat(pad3) + price);
          } else if (cells.length === 4) {
            var c0 = cellTexts[0].padEnd(4);
            var c1 = cellTexts[1];
            var c2 = cellTexts[2];
            var c3 = cellTexts[3];
            var fixedLen = c0.length + c2.length + c3.length + 3;
            var maxC1 = lineWidth - fixedLen;
            if (c1.length > maxC1 && maxC1 > 3) {
              c1 = c1.substring(0, maxC1 - 1) + ".";
            }
            var midPad = Math.max(1, lineWidth - c0.length - c1.length - c2.length - c3.length - 1);
            var halfPad = Math.floor(midPad / 2);
            var restPad = midPad - halfPad;
            lines.push(prefix + c0 + c1 + " ".repeat(halfPad) + c2 + " ".repeat(restPad) + c3);
          } else {
            lines.push(prefix + cellTexts.join("  "));
          }
          return;
        }
      }


      if (
        tag === "table" ||
        tag === "thead" ||
        tag === "tbody" ||
        tag === "tfoot"
      ) {
        for (var ci = 0; ci < node.childNodes.length; ci++) {
          processNode(node.childNodes[ci], context);
        }
        return;
      }

      var blockTags = [
        "div", "p", "h1", "h2", "h3", "h4", "h5", "h6",
        "li", "section", "header", "footer", "address",
      ];

      if (blockTags.includes(tag)) {
        var subContext = { texts: [] };
        for (var bi = 0; bi < node.childNodes.length; bi++) {
          processNode(node.childNodes[bi], subContext);
        }
        var combined = subContext.texts.join(" ").trim();
        if (combined) {
          lines.push(prefix + combined);
        }
        return;
      }

      if (tag === "br") {
        if (context.texts.length > 0) {
          lines.push(prefix + context.texts.join(" ").trim());
          context.texts = [];
        }
        return;
      }

      var inlineText = "";
      for (var ii = 0; ii < node.childNodes.length; ii++) {
        var child = node.childNodes[ii];
        if (child.nodeType === Node.TEXT_NODE) {
          inlineText += child.textContent;
        } else if (child.nodeType === Node.ELEMENT_NODE) {
          if (child.tagName.toLowerCase() === "br") {
            if (inlineText.trim() || context.texts.length > 0) {
              context.texts.push(inlineText.trim());
              lines.push(prefix + context.texts.join(" ").trim());
              context.texts = [];
              inlineText = "";
            }
          } else {
            processNode(child, context);
          }
        }
      }
      if (inlineText.trim()) {
        context.texts.push(inlineText.trim());
      }
    }

    var context = { texts: [] };
    for (var ri = 0; ri < element.childNodes.length; ri++) {
      processNode(element.childNodes[ri], context);
    }
    if (context.texts.length > 0) {
      lines.push(context.texts.join(" ").trim());
    }

    var lastLine = lines[lines.length - 1] || "";
    if (!lastLine.includes("##CUT##")) {
      lines.push("##FEED3##");
      lines.push("##CUT##");
    }

    return lines.join("\n");
  }

  // ---- Check if extension context is still valid ----
  function isExtensionValid() {
    try {
      return !!(chrome.runtime && chrome.runtime.id);
    } catch (e) {
      return false;
    }
  }

  // ---- Send print request to background ----
  var _printInProgress = false;

  function sendPrintRequestWithRetry(printContentText, _retryCount) {
    _retryCount = _retryCount || 0;
    var MAX_RETRIES = 2;

    if (!isExtensionValid()) {
      showNotification(
        "error",
        "Extension Reloaded",
        "Please refresh this page to reconnect to VYROX POS Printer Connector."
      );
      return;
    }

    if (_printInProgress) {
      return;
    }
    _printInProgress = true;

    updateStatus("sending", _retryCount > 0 ? "Retrying..." : "Sending...");

    try {
      chrome.runtime.sendMessage(
        {
          action: "print",
          data: {
            command: "print",
            content: printContentText,
            timestamp: new Date().toISOString(),
          },
        },
        function (response) {
          if (chrome.runtime.lastError) {
            _printInProgress = false;
            if (_retryCount < MAX_RETRIES) {
              setTimeout(function () {
                sendPrintRequestWithRetry(printContentText, _retryCount + 1);
              }, 1500);
              return;
            }
            updateStatus("error", "Extension error");
            showNotification(
              "error",
              "Print Error",
              "Could not connect to VYROX POS Printer Connector. Is the native host installed?"
            );
            setTimeout(function () {
              if (isExtensionValid()) updateStatus("ready", "Ready");
            }, 5000);
            return;
          }

          if (response && response.success) {
            updateStatus("success", "Sent!");
            showNotification("success", "Print Sent", "Content sent to printer.");
            setTimeout(function () {
              if (isExtensionValid()) updateStatus("ready", "Ready");
              _printInProgress = false;
            }, 3000);
          } else {
            var errMsg =
              response && response.error ? response.error : "Unknown error";

            if (errMsg.indexOf("forbidden") !== -1 && _retryCount < MAX_RETRIES) {
              _printInProgress = false;
              setTimeout(function () {
                sendPrintRequestWithRetry(printContentText, _retryCount + 1);
              }, 1500);
              return;
            }

            updateStatus("error", "Failed");
            showNotification("error", "Print Failed", errMsg);
            setTimeout(function () {
              if (isExtensionValid()) updateStatus("ready", "Ready");
              _printInProgress = false;
            }, 5000);
          }
        }
      );
    } catch (e) {
      _printInProgress = false;
      if (_retryCount < MAX_RETRIES) {
        setTimeout(function () {
          sendPrintRequestWithRetry(printContentText, _retryCount + 1);
        }, 1500);
        return;
      }
      showNotification(
        "error",
        "Extension Reloaded",
        "Please refresh this page to reconnect to VYROX POS Printer Connector."
      );
    }
  }

  function sendPrintRequest(printContentText) {
    sendPrintRequestWithRetry(printContentText);
  }

  // ---- Create floating print button ----
  function createFloatingButton() {
    if (floatingBtn) return;

    floatingBtn = document.createElement("div");
    floatingBtn.id = "vyrox-pos-printer-connector-fab";
    floatingBtn.innerHTML =
      '<div class="rp-fab-btn" title="Print via VYROX POS Printer Connector">' +
      '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" ' +
      'viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" ' +
      'stroke-linecap="round" stroke-linejoin="round">' +
      '<polyline points="6 9 6 2 18 2 18 9"></polyline>' +
      '<path d="M6 18H4a2 2 0 0 1-2-2v-5a2 2 0 0 1 2-2h16a2 2 0 0 1 2 2v5a2 2 0 0 1-2 2h-2"></path>' +
      '<rect x="6" y="14" width="12" height="8"></rect>' +
      "</svg>" +
      "</div>" +
      '<div class="rp-fab-status" id="rp-status">Ready</div>';

    floatingBtn.addEventListener("click", handlePrintClick);
    document.body.appendChild(floatingBtn);

    statusIndicator = document.getElementById("rp-status");
  }

  // ---- Handle print button click ----
  function handlePrintClick() {
    var printContentEl = findPrintContentElement();

    if (!printContentEl) {
      showNotification(
        "error",
        "No Print Content Found",
        "Could not find a printable element on this page. " +
          'Add id="vyrox_pos_printer_connector" or class="vyrox_pos_printer_connector" to your print container.'
      );
      return;
    }

    var printContentText = extractPrintContentText(printContentEl);

    if (!printContentText || printContentText.trim().length === 0) {
      showNotification("error", "Empty Content", "Print content is empty.");
      return;
    }

    sendPrintRequest(printContentText);
  }

  // ---- Update status indicator ----
  function updateStatus(state, text) {
    if (!statusIndicator) return;

    statusIndicator.textContent = text;
    statusIndicator.className = "rp-fab-status";

    var stateMap = {
      ready: "rp-status-ready",
      sending: "rp-status-sending",
      success: "rp-status-success",
      error: "rp-status-error",
    };

    if (stateMap[state]) {
      statusIndicator.classList.add(stateMap[state]);
    }
  }

  // ---- Show notification toast ----
  function showNotification(type, title, message) {
    var existing = document.getElementById("rp-notification");
    if (existing) existing.remove();

    var toast = document.createElement("div");
    toast.id = "rp-notification";
    toast.className = "rp-toast rp-toast-" + type;

    var iconSvg =
      type === "success"
        ? '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path><polyline points="22 4 12 14.01 9 11.01"></polyline></svg>'
        : '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"></circle><line x1="15" y1="9" x2="9" y2="15"></line><line x1="9" y1="9" x2="15" y2="15"></line></svg>';

    toast.innerHTML =
      '<div class="rp-toast-icon">' +
      iconSvg +
      "</div>" +
      '<div class="rp-toast-body">' +
      '<div class="rp-toast-title">' +
      escapeHtml(title) +
      "</div>" +
      '<div class="rp-toast-message">' +
      escapeHtml(message) +
      "</div>" +
      "</div>" +
      '<button class="rp-toast-close" id="rp-toast-close-btn">' +
      '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="18" y1="6" x2="6" y2="18"></line><line x1="6" y1="6" x2="18" y2="18"></line></svg>' +
      "</button>";

    document.body.appendChild(toast);

    var closeBtnEl = document.getElementById("rp-toast-close-btn");
    if (closeBtnEl) {
      closeBtnEl.addEventListener("click", function () {
        toast.remove();
      });
    }

    setTimeout(function () {
      if (toast.parentElement) {
        toast.classList.add("rp-toast-fadeout");
        setTimeout(function () {
          toast.remove();
        }, 300);
      }
    }, 6000);
  }

  // ---- Escape HTML to prevent XSS in toast messages ----
  function escapeHtml(text) {
    var div = document.createElement("div");
    div.appendChild(document.createTextNode(text));
    return div.innerHTML;
  }

  // ---- Intercept existing print buttons ----
  function interceptExistingButtons() {
    for (var s = 0; s < EXISTING_PRINT_BTN_SELECTORS.length; s++) {
      var btns = document.querySelectorAll(EXISTING_PRINT_BTN_SELECTORS[s]);
      btns.forEach(function (btn) {
        if (btn.hasAttribute("data-vyrox-intercepted")) return;
        btn.setAttribute("data-vyrox-intercepted", "true");
        btn.addEventListener(
          "click",
          function (e) {
            var printContentEl = findPrintContentElement();
            if (printContentEl) {
              e.preventDefault();
              e.stopPropagation();
              e.stopImmediatePropagation();
              handlePrintClick();
            }
          },
          true
        );
      });
    }
  }

  // ---- Listen for custom print events from page ----
  window.addEventListener("message", function (event) {
    if (event.source !== window) return;

    if (event.data && event.data.type === "VYROX_POS_PRINTER_CONNECTOR_PRINT") {
      var content = event.data.content;
      if (content) {
        sendPrintRequestWithRetry(content);
      }
    }
  });

  // ---- Listen for popup trigger ----
  try {
    chrome.runtime.onMessage.addListener(function (
      message,
      sender,
      sendResponse
    ) {
      if (message.action === "triggerPrint") {
        var printContentEl = findPrintContentElement();

        if (!printContentEl) {
          sendResponse({
            success: false,
            error: "No print content element found on this page",
          });
          return;
        }

        var text = extractPrintContentText(printContentEl);

        if (!text || text.trim().length === 0) {
          sendResponse({ success: false, error: "Print content is empty" });
          return;
        }

        sendPrintRequest(text);
        sendResponse({ success: true });
      }
    });
  } catch (e) {}

  // ---- Initialize ----
  function init() {
    injectStyles();

    var printContentEl = findPrintContentElement();
    if (printContentEl) {
      createFloatingButton();
      interceptExistingButtons();
    }

    var observer = new MutationObserver(function () {
      if (!isExtensionValid()) {
        observer.disconnect();
        return;
      }
      var el = findPrintContentElement();
      if (el && !floatingBtn) {
        createFloatingButton();
        interceptExistingButtons();
      }
    });

    observer.observe(document.body, { childList: true, subtree: true });
  }

  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", init);
  } else {
    init();
  }
})();